/**
 * BossOfToys Manager - Dashboard JavaScript
 */

(function($) {
    'use strict';

    // Zmienne globalne
    let eventSource = null;
    let currentJobId = null;
    let isConnected = false;

    // Nazwy modułów po polsku
    const moduleNames = {
        'stock-sync': 'Synchronizacja stanów',
        'product-adder': 'Dodawanie produktów',
        'product-deleter': 'Usuwanie produktów',
        'price-updater': 'Aktualizacja cen',
        'order-forwarder': 'Przekazywanie zamówień',
        'description-generator': 'Generator opisów AI'
    };

    // Statusy po polsku
    const statusNames = {
        'queued': 'W kolejce',
        'running': 'W toku',
        'completed': 'Zakończono',
        'failed': 'Błąd',
        'cancelled': 'Anulowano'
    };

    // ========== Inicjalizacja ==========

    $(document).ready(function() {
        // Dashboard
        if ($('.bot-modules-grid').length) {
            initDashboard();
        }

        // Ustawienia
        if ($('#bot-save-settings').length) {
            initSettings();
        }

        // Harmonogram
        if ($('#bot-schedule-form').length) {
            initSchedule();
        }
    });

    // ========== Dashboard ==========

    function initDashboard() {
        // Pobierz początkowy status
        refreshStatus();

        // Uruchom SSE jeśli skonfigurowane
        if (botData.apiConfigured && botData.sseUrl) {
            connectSSE();
        }

        // Event: Uruchom moduł
        $('.bot-run-btn').on('click', function() {
            const module = $(this).data('module');
            const card = $(this).closest('.bot-module-card');
            const dryRun = card.find('.bot-dry-run').is(':checked');

            if (confirm(botData.strings.confirmRun)) {
                runModule(module, { dry_run: dryRun });
            }
        });

        // Event: Anuluj zadanie
        $('#bot-cancel-job').on('click', function() {
            if (currentJobId && confirm(botData.strings.confirmCancel)) {
                cancelJob(currentJobId);
            }
        });

        // Event: Wyczyść logi
        $('#bot-clear-logs').on('click', function() {
            $('#bot-logs').empty();
        });

        // Odśwież status co 10 sekund (fallback jeśli SSE nie działa)
        setInterval(refreshStatus, 10000);
    }

    // ========== SSE (Server-Sent Events) ==========

    function connectSSE() {
        if (eventSource) {
            eventSource.close();
        }

        eventSource = new EventSource(botData.sseUrl);

        eventSource.onopen = function() {
            isConnected = true;
            updateStatusIndicator(true);
        };

        eventSource.onerror = function() {
            isConnected = false;
            updateStatusIndicator(false);
        };

        eventSource.addEventListener('status', function(e) {
            const data = JSON.parse(e.data);
            handleStatusUpdate(data);
        });

        eventSource.addEventListener('log', function(e) {
            const data = JSON.parse(e.data);
            appendLog(data);
        });

        eventSource.addEventListener('finished', function(e) {
            const data = JSON.parse(e.data);
            handleJobFinished(data);
        });
    }

    function updateStatusIndicator(connected) {
        const dot = $('#bot-status-dot');
        const text = $('#bot-status-text');

        if (connected) {
            dot.removeClass('disconnected').addClass('connected');
            text.text('Połączono');
        } else {
            dot.removeClass('connected').addClass('disconnected');
            text.text('Rozłączono');
        }
    }

    function handleStatusUpdate(data) {
        // Aktualizuj kolejkę
        $('#bot-queue-count').text(data.queue_length || 0);

        // Aktualizuj aktywne zadanie
        if (data.running) {
            currentJobId = data.running.job_id;
            showActiveJob(data.running);
        } else {
            currentJobId = null;
            hideActiveJob();
        }

        // Aktualizuj historię
        if (data.recent_jobs) {
            updateHistoryTable(data.recent_jobs);
        }
    }

    function handleJobFinished(data) {
        // Pokaż powiadomienie
        const status = statusNames[data.status] || data.status;
        const message = data.summary || status;

        appendLog({
            level: data.status === 'completed' ? 'success' : 'error',
            message: `Zadanie zakończone: ${message}`,
            timestamp: new Date().toISOString()
        });

        // Odśwież status
        setTimeout(refreshStatus, 1000);
    }

    // ========== Zadania ==========

    function runModule(module, params) {
        const btn = $(`.bot-run-btn[data-module="${module}"]`);
        btn.prop('disabled', true).find('.dashicons').removeClass('dashicons-controls-play').addClass('dashicons-update spin');

        $.ajax({
            url: botData.ajaxUrl,
            method: 'POST',
            data: {
                action: 'bot_run_module',
                nonce: botData.nonce,
                module: module,
                params: params
            },
            success: function(response) {
                if (response.success) {
                    currentJobId = response.data.job_id;
                    showActiveJob(response.data);
                    appendLog({
                        level: 'info',
                        message: `Uruchomiono: ${moduleNames[module] || module}`,
                        timestamp: new Date().toISOString()
                    });
                } else {
                    alert(response.data.message || 'Błąd uruchamiania modułu');
                }
            },
            error: function() {
                alert(botData.strings.connectionError);
            },
            complete: function() {
                btn.prop('disabled', false).find('.dashicons').removeClass('dashicons-update spin').addClass('dashicons-controls-play');
            }
        });
    }

    function cancelJob(jobId) {
        $.ajax({
            url: botData.ajaxUrl,
            method: 'POST',
            data: {
                action: 'bot_cancel_job',
                nonce: botData.nonce,
                job_id: jobId
            },
            success: function(response) {
                if (response.success) {
                    appendLog({
                        level: 'warning',
                        message: 'Zadanie anulowane',
                        timestamp: new Date().toISOString()
                    });
                } else {
                    alert(response.data.message || 'Błąd anulowania');
                }
            },
            error: function() {
                alert(botData.strings.connectionError);
            }
        });
    }

    function refreshStatus() {
        if (!botData.apiConfigured) return;

        $.ajax({
            url: botData.ajaxUrl,
            method: 'POST',
            data: {
                action: 'bot_get_status',
                nonce: botData.nonce
            },
            success: function(response) {
                if (response.success) {
                    handleStatusUpdate({
                        running: response.data.current_job,
                        queue_length: response.data.queue_length
                    });

                    // Aktualizuj historię z jobs listy
                    if (response.data.stats && response.data.stats.last_runs) {
                        // Możemy pobrać pełną listę jobów osobno
                    }
                }
            }
        });
    }

    // ========== UI Helpers ==========

    function showActiveJob(job) {
        const container = $('#bot-active-job');
        container.show();

        $('#bot-job-module').text(moduleNames[job.module] || job.module);
        $('#bot-job-status').text(statusNames[job.status] || job.status);

        const progress = Math.round((job.progress || 0) * 100);
        $('#bot-progress-fill').css('width', progress + '%');
        $('#bot-progress-text').text(progress + '%');
    }

    function hideActiveJob() {
        $('#bot-active-job').hide();
    }

    function appendLog(log) {
        const container = $('#bot-logs');
        const time = new Date(log.timestamp).toLocaleTimeString('pl-PL');

        let levelClass = 'log-info';
        if (log.level === 'error') levelClass = 'log-error';
        else if (log.level === 'warning') levelClass = 'log-warning';
        else if (log.level === 'success') levelClass = 'log-success';

        const html = `
            <div class="bot-log-entry ${levelClass}">
                <span class="bot-log-time">[${time}]</span>
                <span class="bot-log-message">${escapeHtml(log.message)}</span>
            </div>
        `;

        container.append(html);
        container.scrollTop(container[0].scrollHeight);
    }

    function updateHistoryTable(jobs) {
        const tbody = $('#bot-history-body');
        tbody.empty();

        if (!jobs || jobs.length === 0) {
            tbody.html('<tr><td colspan="5">Brak zadań</td></tr>');
            return;
        }

        jobs.forEach(function(job) {
            const date = new Date(job.created_at).toLocaleString('pl-PL');
            const status = statusNames[job.status] || job.status;
            const module = moduleNames[job.module] || job.module;

            let statusClass = 'status-default';
            if (job.status === 'completed') statusClass = 'status-success';
            else if (job.status === 'failed') statusClass = 'status-error';
            else if (job.status === 'running') statusClass = 'status-running';

            const html = `
                <tr>
                    <td><code>${job.job_id}</code></td>
                    <td>${escapeHtml(module)}</td>
                    <td><span class="bot-status-badge ${statusClass}">${status}</span></td>
                    <td>${date}</td>
                    <td>${escapeHtml(job.summary || '-')}</td>
                </tr>
            `;
            tbody.append(html);
        });
    }

    function escapeHtml(text) {
        if (!text) return '';
        const div = document.createElement('div');
        div.textContent = text;
        return div.innerHTML;
    }

    // ========== Ustawienia ==========

    function initSettings() {
        // Toggle password visibility
        $('.bot-toggle-password').on('click', function() {
            const target = $(this).data('target');
            const input = $('#' + target);
            const icon = $(this).find('.dashicons');

            if (input.attr('type') === 'password') {
                input.attr('type', 'text');
                icon.removeClass('dashicons-visibility').addClass('dashicons-hidden');
            } else {
                input.attr('type', 'password');
                icon.removeClass('dashicons-hidden').addClass('dashicons-visibility');
            }
        });

        // Test połączenia
        $('#bot-test-connection').on('click', function() {
            const btn = $(this);
            const spinner = $('#bot-settings-spinner');
            const message = $('#bot-settings-message');

            const apiUrl = $('#bot-api-url').val();
            const apiKey = $('#bot-api-key').val();

            if (!apiUrl || !apiKey) {
                message.removeClass('success').addClass('error').text('Podaj URL i klucz API');
                return;
            }

            btn.prop('disabled', true);
            spinner.addClass('is-active');
            message.text('');

            $.ajax({
                url: botData.ajaxUrl,
                method: 'POST',
                data: {
                    action: 'bot_test_connection',
                    nonce: botData.nonce,
                    api_url: apiUrl,
                    api_key: apiKey
                },
                success: function(response) {
                    if (response.success) {
                        message.removeClass('error').addClass('success').text(response.data.message);
                    } else {
                        message.removeClass('success').addClass('error').text(response.data.message);
                    }
                },
                error: function() {
                    message.removeClass('success').addClass('error').text('Błąd połączenia');
                },
                complete: function() {
                    btn.prop('disabled', false);
                    spinner.removeClass('is-active');
                }
            });
        });

        // Zapisz ustawienia
        $('#bot-save-settings').on('click', function() {
            const btn = $(this);
            const spinner = $('#bot-settings-spinner');
            const message = $('#bot-settings-message');

            const apiUrl = $('#bot-api-url').val();
            const apiKey = $('#bot-api-key').val();

            btn.prop('disabled', true);
            spinner.addClass('is-active');
            message.text('');

            $.ajax({
                url: botData.ajaxUrl,
                method: 'POST',
                data: {
                    action: 'bot_save_settings',
                    nonce: botData.nonce,
                    api_url: apiUrl,
                    api_key: apiKey
                },
                success: function(response) {
                    if (response.success) {
                        message.removeClass('error').addClass('success').text(response.data.message);
                        // Odśwież stronę po 1s
                        setTimeout(function() {
                            location.reload();
                        }, 1000);
                    } else {
                        message.removeClass('success').addClass('error').text(response.data.message);
                    }
                },
                error: function() {
                    message.removeClass('success').addClass('error').text('Błąd zapisywania');
                },
                complete: function() {
                    btn.prop('disabled', false);
                    spinner.removeClass('is-active');
                }
            });
        });
    }

    // ========== Harmonogram ==========

    function initSchedule() {
        // Toggle enabled state visual
        $('.bot-schedule-card input[type="checkbox"]').on('change', function() {
            const card = $(this).closest('.bot-schedule-card');
            if ($(this).is(':checked')) {
                card.addClass('bot-enabled');
            } else {
                card.removeClass('bot-enabled');
            }
        });

        // Submit form
        $('#bot-schedule-form').on('submit', function(e) {
            e.preventDefault();

            const spinner = $('#bot-schedule-spinner');
            const message = $('#bot-schedule-message');
            const formData = $(this).serializeArray();

            // Przekształć dane formularza
            const schedule = {};
            formData.forEach(function(item) {
                const match = item.name.match(/schedule\[(\w+)\]\[(\w+)\]/);
                if (match) {
                    const module = match[1];
                    const key = match[2];
                    if (!schedule[module]) {
                        schedule[module] = { enabled: false };
                    }
                    schedule[module][key] = item.value;
                }
            });

            // Upewnij się, że wszystkie moduły mają enabled ustawione
            $('.bot-schedule-card').each(function() {
                const checkbox = $(this).find('input[type="checkbox"]');
                const name = checkbox.attr('name');
                const match = name.match(/schedule\[(\w+)\]/);
                if (match) {
                    const module = match[1];
                    if (!schedule[module]) {
                        schedule[module] = {};
                    }
                    schedule[module].enabled = checkbox.is(':checked');
                    const select = $(this).find('select');
                    schedule[module].interval = select.val();
                }
            });

            spinner.addClass('is-active');
            message.text('');

            $.ajax({
                url: botData.ajaxUrl,
                method: 'POST',
                data: {
                    action: 'bot_save_schedule',
                    nonce: botData.nonce,
                    schedule: schedule
                },
                success: function(response) {
                    if (response.success) {
                        message.removeClass('error').addClass('success').text(response.data.message);
                    } else {
                        message.removeClass('success').addClass('error').text(response.data.message);
                    }
                },
                error: function() {
                    message.removeClass('success').addClass('error').text('Błąd zapisywania');
                },
                complete: function() {
                    spinner.removeClass('is-active');
                }
            });
        });
    }

})(jQuery);
