<?php
/**
 * Plugin Name: BossOfToys Manager
 * Plugin URI: https://github.com/RadoslawSwider/RSJB-Bossofotys
 * Description: Panel zarządzania integracją BossOfToys x WooCommerce. Pozwala na zdalne wyzwalanie synchronizacji produktów, stanów magazynowych, cen i zamówień.
 * Version: 1.0.0
 * Author: RSJB
 * Author URI: https://github.com/RadoslawSwider
 * License: GPL v2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: bossoftoys-manager
 * Domain Path: /languages
 * Requires at least: 5.8
 * Requires PHP: 7.4
 * WC requires at least: 6.0
 * WC tested up to: 8.5
 */

// Zabezpieczenie przed bezpośrednim dostępem
defined('ABSPATH') || exit;

// Stałe pluginu
define('BOT_VERSION', '1.0.0');
define('BOT_PLUGIN_FILE', __FILE__);
define('BOT_PLUGIN_DIR', plugin_dir_path(__FILE__));
define('BOT_PLUGIN_URL', plugin_dir_url(__FILE__));
define('BOT_PLUGIN_BASENAME', plugin_basename(__FILE__));

/**
 * Główna klasa pluginu
 */
final class BossOfToys_Manager {

    /**
     * Instancja singletona
     */
    private static $instance = null;

    /**
     * Instancja klasy API
     */
    public $api = null;

    /**
     * Instancja klasy Admin
     */
    public $admin = null;

    /**
     * Instancja klasy Cron
     */
    public $cron = null;

    /**
     * Pobierz instancję singletona
     */
    public static function instance() {
        if (is_null(self::$instance)) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    /**
     * Konstruktor
     */
    private function __construct() {
        $this->includes();
        $this->init_hooks();
    }

    /**
     * Załaduj pliki
     */
    private function includes() {
        // Klasy podstawowe
        require_once BOT_PLUGIN_DIR . 'includes/class-bot-api.php';
        require_once BOT_PLUGIN_DIR . 'includes/class-bot-admin.php';
        require_once BOT_PLUGIN_DIR . 'includes/class-bot-cron.php';
    }

    /**
     * Inicjalizuj hooki
     */
    private function init_hooks() {
        // Aktywacja/Deaktywacja
        register_activation_hook(BOT_PLUGIN_FILE, array($this, 'activate'));
        register_deactivation_hook(BOT_PLUGIN_FILE, array($this, 'deactivate'));

        // Inicjalizacja po załadowaniu pluginów
        add_action('plugins_loaded', array($this, 'init'), 0);

        // Linki w liście pluginów
        add_filter('plugin_action_links_' . BOT_PLUGIN_BASENAME, array($this, 'plugin_action_links'));
    }

    /**
     * Inicjalizacja pluginu
     */
    public function init() {
        // Sprawdź czy WooCommerce jest aktywny
        if (!class_exists('WooCommerce')) {
            add_action('admin_notices', array($this, 'woocommerce_missing_notice'));
            return;
        }

        // Inicjalizuj komponenty
        $this->api = new BOT_API();
        $this->admin = new BOT_Admin();
        $this->cron = new BOT_Cron();

        // Załaduj tłumaczenia
        load_plugin_textdomain('bossoftoys-manager', false, dirname(BOT_PLUGIN_BASENAME) . '/languages');
    }

    /**
     * Aktywacja pluginu
     */
    public function activate() {
        // Utwórz domyślne opcje
        $default_options = array(
            'api_url' => '',
            'api_key' => '',
            'connected' => false,
            'last_connection_test' => null,
        );

        if (!get_option('bot_settings')) {
            add_option('bot_settings', $default_options);
        }

        // Utwórz opcje schedulera
        $default_schedule = array(
            'stock_sync' => array(
                'enabled' => false,
                'interval' => 'every_30_minutes',
            ),
            'price_updater' => array(
                'enabled' => false,
                'interval' => 'daily',
            ),
            'order_forwarder' => array(
                'enabled' => false,
                'interval' => 'every_15_minutes',
            ),
        );

        if (!get_option('bot_schedule')) {
            add_option('bot_schedule', $default_schedule);
        }

        // Wyczyść cache
        flush_rewrite_rules();
    }

    /**
     * Deaktywacja pluginu
     */
    public function deactivate() {
        // Usuń zaplanowane zadania cron
        wp_clear_scheduled_hook('bot_cron_stock_sync');
        wp_clear_scheduled_hook('bot_cron_price_updater');
        wp_clear_scheduled_hook('bot_cron_order_forwarder');
        wp_clear_scheduled_hook('bot_cron_product_adder');
        wp_clear_scheduled_hook('bot_cron_product_deleter');
        wp_clear_scheduled_hook('bot_cron_description_generator');

        // Wyczyść cache
        flush_rewrite_rules();
    }

    /**
     * Linki akcji w liście pluginów
     */
    public function plugin_action_links($links) {
        $plugin_links = array(
            '<a href="' . admin_url('admin.php?page=bossoftoys-manager') . '">' . __('Dashboard', 'bossoftoys-manager') . '</a>',
            '<a href="' . admin_url('admin.php?page=bossoftoys-settings') . '">' . __('Ustawienia', 'bossoftoys-manager') . '</a>',
        );
        return array_merge($plugin_links, $links);
    }

    /**
     * Powiadomienie o braku WooCommerce
     */
    public function woocommerce_missing_notice() {
        ?>
        <div class="notice notice-error">
            <p>
                <strong>BossOfToys Manager</strong> wymaga aktywnego pluginu
                <a href="<?php echo admin_url('plugin-install.php?s=woocommerce&tab=search&type=term'); ?>">WooCommerce</a>.
            </p>
        </div>
        <?php
    }
}

/**
 * Funkcja pomocnicza do pobierania instancji pluginu
 */
function BOT() {
    return BossOfToys_Manager::instance();
}

// Uruchom plugin
BOT();
