<?php
/**
 * Klasa panelu administracyjnego
 *
 * @package BossOfToys_Manager
 */

defined('ABSPATH') || exit;

/**
 * Klasa BOT_Admin
 */
class BOT_Admin {

    /**
     * Konstruktor
     */
    public function __construct() {
        // Menu admina
        add_action('admin_menu', array($this, 'add_admin_menu'));

        // Skrypty i style
        add_action('admin_enqueue_scripts', array($this, 'enqueue_assets'));

        // AJAX handlers
        add_action('wp_ajax_bot_test_connection', array($this, 'ajax_test_connection'));
        add_action('wp_ajax_bot_save_settings', array($this, 'ajax_save_settings'));
        add_action('wp_ajax_bot_run_module', array($this, 'ajax_run_module'));
        add_action('wp_ajax_bot_cancel_job', array($this, 'ajax_cancel_job'));
        add_action('wp_ajax_bot_get_status', array($this, 'ajax_get_status'));
        add_action('wp_ajax_bot_get_job_logs', array($this, 'ajax_get_job_logs'));
        add_action('wp_ajax_bot_save_schedule', array($this, 'ajax_save_schedule'));
    }

    /**
     * Dodaj menu w panelu admina
     */
    public function add_admin_menu() {
        // Główne menu
        add_menu_page(
            __('BossOfToys Manager', 'bossoftoys-manager'),
            __('BossOfToys', 'bossoftoys-manager'),
            'manage_woocommerce',
            'bossoftoys-manager',
            array($this, 'render_dashboard'),
            'dashicons-update',
            56
        );

        // Dashboard (duplikat głównego menu)
        add_submenu_page(
            'bossoftoys-manager',
            __('Dashboard', 'bossoftoys-manager'),
            __('Dashboard', 'bossoftoys-manager'),
            'manage_woocommerce',
            'bossoftoys-manager',
            array($this, 'render_dashboard')
        );

        // Harmonogram
        add_submenu_page(
            'bossoftoys-manager',
            __('Harmonogram', 'bossoftoys-manager'),
            __('Harmonogram', 'bossoftoys-manager'),
            'manage_woocommerce',
            'bossoftoys-schedule',
            array($this, 'render_schedule')
        );

        // Ustawienia
        add_submenu_page(
            'bossoftoys-manager',
            __('Ustawienia', 'bossoftoys-manager'),
            __('Ustawienia', 'bossoftoys-manager'),
            'manage_woocommerce',
            'bossoftoys-settings',
            array($this, 'render_settings')
        );
    }

    /**
     * Załaduj skrypty i style
     */
    public function enqueue_assets($hook) {
        // Tylko na stronach pluginu
        if (strpos($hook, 'bossoftoys') === false) {
            return;
        }

        // CSS
        wp_enqueue_style(
            'bot-admin-css',
            BOT_PLUGIN_URL . 'admin/css/admin.css',
            array(),
            BOT_VERSION
        );

        // JavaScript
        wp_enqueue_script(
            'bot-admin-js',
            BOT_PLUGIN_URL . 'admin/js/dashboard.js',
            array('jquery'),
            BOT_VERSION,
            true
        );

        // Przekaż dane do JS
        $settings = get_option('bot_settings', array());
        $api = BOT()->api;

        wp_localize_script('bot-admin-js', 'botData', array(
            'ajaxUrl'        => admin_url('admin-ajax.php'),
            'nonce'          => wp_create_nonce('bot_ajax_nonce'),
            'apiConfigured'  => $api->is_configured(),
            'sseUrl'         => $api->is_configured() ? $api->get_sse_dashboard_url() : '',
            'strings'        => array(
                'confirmCancel'   => __('Czy na pewno chcesz anulować to zadanie?', 'bossoftoys-manager'),
                'confirmRun'      => __('Czy na pewno chcesz uruchomić ten moduł?', 'bossoftoys-manager'),
                'running'         => __('W toku...', 'bossoftoys-manager'),
                'completed'       => __('Zakończono', 'bossoftoys-manager'),
                'failed'          => __('Błąd', 'bossoftoys-manager'),
                'cancelled'       => __('Anulowano', 'bossoftoys-manager'),
                'connectionError' => __('Błąd połączenia z API', 'bossoftoys-manager'),
            ),
        ));
    }

    /**
     * Renderuj Dashboard
     */
    public function render_dashboard() {
        include BOT_PLUGIN_DIR . 'admin/views/dashboard.php';
    }

    /**
     * Renderuj Harmonogram
     */
    public function render_schedule() {
        include BOT_PLUGIN_DIR . 'admin/views/schedule.php';
    }

    /**
     * Renderuj Ustawienia
     */
    public function render_settings() {
        include BOT_PLUGIN_DIR . 'admin/views/settings.php';
    }

    // ========== AJAX Handlers ==========

    /**
     * AJAX: Test połączenia
     */
    public function ajax_test_connection() {
        check_ajax_referer('bot_ajax_nonce', 'nonce');

        if (!current_user_can('manage_woocommerce')) {
            wp_send_json_error(array('message' => 'Brak uprawnień'));
        }

        // Tymczasowo użyj przesłanych danych do testu
        $api_url = sanitize_text_field($_POST['api_url'] ?? '');
        $api_key = sanitize_text_field($_POST['api_key'] ?? '');

        if (empty($api_url) || empty($api_key)) {
            wp_send_json_error(array('message' => 'Podaj URL i klucz API'));
        }

        // Wykonaj test
        $url = rtrim($api_url, '/') . '/health';

        $response = wp_remote_get($url, array(
            'timeout' => 10,
            'headers' => array(
                'X-API-Key' => $api_key,
            ),
        ));

        if (is_wp_error($response)) {
            wp_send_json_error(array('message' => $response->get_error_message()));
        }

        $body = json_decode(wp_remote_retrieve_body($response), true);
        $status_code = wp_remote_retrieve_response_code($response);

        if ($status_code === 200 && isset($body['status']) && $body['status'] === 'healthy') {
            wp_send_json_success(array('message' => 'Połączono z API'));
        } else {
            $error = $body['error'] ?? $body['detail'] ?? 'Nieoczekiwana odpowiedź (kod: ' . $status_code . ')';
            wp_send_json_error(array('message' => $error));
        }
    }

    /**
     * AJAX: Zapisz ustawienia
     */
    public function ajax_save_settings() {
        check_ajax_referer('bot_ajax_nonce', 'nonce');

        if (!current_user_can('manage_woocommerce')) {
            wp_send_json_error(array('message' => 'Brak uprawnień'));
        }

        $api_url = sanitize_text_field($_POST['api_url'] ?? '');
        $api_key = sanitize_text_field($_POST['api_key'] ?? '');

        $settings = get_option('bot_settings', array());
        $settings['api_url'] = rtrim($api_url, '/');
        $settings['api_key'] = $api_key;
        $settings['connected'] = false; // Reset - wymaga testu

        update_option('bot_settings', $settings);

        wp_send_json_success(array('message' => 'Ustawienia zapisane'));
    }

    /**
     * AJAX: Uruchom moduł
     */
    public function ajax_run_module() {
        check_ajax_referer('bot_ajax_nonce', 'nonce');

        if (!current_user_can('manage_woocommerce')) {
            wp_send_json_error(array('message' => 'Brak uprawnień'));
        }

        $module = sanitize_text_field($_POST['module'] ?? '');
        $params = isset($_POST['params']) ? (array) $_POST['params'] : array();

        // Sanityzacja parametrów
        $sanitized_params = array();
        foreach ($params as $key => $value) {
            $key = sanitize_key($key);
            if (is_bool($value) || $value === 'true' || $value === 'false') {
                $sanitized_params[$key] = filter_var($value, FILTER_VALIDATE_BOOLEAN);
            } elseif (is_numeric($value)) {
                $sanitized_params[$key] = intval($value);
            } else {
                $sanitized_params[$key] = sanitize_text_field($value);
            }
        }

        $api = BOT()->api;
        $result = $api->run_module($module, $sanitized_params);

        if (is_wp_error($result)) {
            wp_send_json_error(array('message' => $result->get_error_message()));
        }

        wp_send_json_success($result);
    }

    /**
     * AJAX: Anuluj zadanie
     */
    public function ajax_cancel_job() {
        check_ajax_referer('bot_ajax_nonce', 'nonce');

        if (!current_user_can('manage_woocommerce')) {
            wp_send_json_error(array('message' => 'Brak uprawnień'));
        }

        $job_id = sanitize_text_field($_POST['job_id'] ?? '');

        $api = BOT()->api;
        $result = $api->cancel_job($job_id);

        if (is_wp_error($result)) {
            wp_send_json_error(array('message' => $result->get_error_message()));
        }

        wp_send_json_success($result);
    }

    /**
     * AJAX: Pobierz status
     */
    public function ajax_get_status() {
        check_ajax_referer('bot_ajax_nonce', 'nonce');

        if (!current_user_can('manage_woocommerce')) {
            wp_send_json_error(array('message' => 'Brak uprawnień'));
        }

        $api = BOT()->api;
        $result = $api->get_status();

        if (is_wp_error($result)) {
            wp_send_json_error(array('message' => $result->get_error_message()));
        }

        wp_send_json_success($result);
    }

    /**
     * AJAX: Pobierz logi zadania
     */
    public function ajax_get_job_logs() {
        check_ajax_referer('bot_ajax_nonce', 'nonce');

        if (!current_user_can('manage_woocommerce')) {
            wp_send_json_error(array('message' => 'Brak uprawnień'));
        }

        $job_id = sanitize_text_field($_POST['job_id'] ?? '');

        $api = BOT()->api;
        $result = $api->get_job_logs($job_id);

        if (is_wp_error($result)) {
            wp_send_json_error(array('message' => $result->get_error_message()));
        }

        wp_send_json_success($result);
    }

    /**
     * AJAX: Zapisz harmonogram
     */
    public function ajax_save_schedule() {
        check_ajax_referer('bot_ajax_nonce', 'nonce');

        if (!current_user_can('manage_woocommerce')) {
            wp_send_json_error(array('message' => 'Brak uprawnień'));
        }

        $schedule = isset($_POST['schedule']) ? (array) $_POST['schedule'] : array();

        // Sanityzacja
        $sanitized = array();
        foreach ($schedule as $module => $config) {
            $module = sanitize_key($module);
            $sanitized[$module] = array(
                'enabled'  => isset($config['enabled']) && filter_var($config['enabled'], FILTER_VALIDATE_BOOLEAN),
                'interval' => sanitize_text_field($config['interval'] ?? 'daily'),
            );
        }

        update_option('bot_schedule', $sanitized);

        // Przeładuj harmonogram cron
        if (BOT()->cron) {
            BOT()->cron->reschedule_all();
        }

        wp_send_json_success(array('message' => 'Harmonogram zapisany'));
    }
}
