<?php
/**
 * Klasa do komunikacji z Python API
 *
 * @package BossOfToys_Manager
 */

defined('ABSPATH') || exit;

/**
 * Klasa BOT_API
 */
class BOT_API {

    /**
     * URL API
     */
    private $api_url = '';

    /**
     * Klucz API
     */
    private $api_key = '';

    /**
     * Timeout dla requestów (sekundy)
     */
    private $timeout = 30;

    /**
     * Konstruktor
     */
    public function __construct() {
        $settings = get_option('bot_settings', array());
        $this->api_url = rtrim($settings['api_url'] ?? '', '/');
        $this->api_key = $settings['api_key'] ?? '';
    }

    /**
     * Sprawdź czy API jest skonfigurowane
     */
    public function is_configured() {
        return !empty($this->api_url) && !empty($this->api_key);
    }

    /**
     * Wykonaj request do API
     *
     * @param string $endpoint Endpoint (np. /api/status)
     * @param string $method   Metoda HTTP (GET, POST, PUT, DELETE)
     * @param array  $data     Dane do wysłania (dla POST/PUT)
     * @return array|WP_Error
     */
    public function request($endpoint, $method = 'GET', $data = null) {
        if (!$this->is_configured()) {
            return new WP_Error('not_configured', 'API nie jest skonfigurowane. Przejdź do Ustawień.');
        }

        $url = $this->api_url . $endpoint;

        $args = array(
            'method'  => $method,
            'timeout' => $this->timeout,
            'headers' => array(
                'X-API-Key'    => $this->api_key,
                'Content-Type' => 'application/json',
                'Accept'       => 'application/json',
            ),
        );

        if ($data !== null && in_array($method, array('POST', 'PUT'))) {
            $args['body'] = json_encode($data);
        }

        $response = wp_remote_request($url, $args);

        if (is_wp_error($response)) {
            return $response;
        }

        $status_code = wp_remote_retrieve_response_code($response);
        $body = wp_remote_retrieve_body($response);

        // Parsuj JSON
        $decoded = json_decode($body, true);

        if ($status_code >= 400) {
            $error_message = $decoded['error'] ?? $decoded['detail'] ?? 'Nieznany błąd API';
            return new WP_Error('api_error', $error_message, array('status' => $status_code));
        }

        return $decoded;
    }

    // ========== Status ==========

    /**
     * Pobierz status API
     */
    public function get_status() {
        return $this->request('/api/status');
    }

    /**
     * Test połączenia z API
     */
    public function test_connection() {
        $result = $this->request('/health');

        if (is_wp_error($result)) {
            return array(
                'success' => false,
                'message' => $result->get_error_message(),
            );
        }

        if (isset($result['status']) && $result['status'] === 'healthy') {
            // Zaktualizuj opcję połączenia
            $settings = get_option('bot_settings', array());
            $settings['connected'] = true;
            $settings['last_connection_test'] = current_time('mysql');
            update_option('bot_settings', $settings);

            return array(
                'success' => true,
                'message' => 'Połączono z API',
            );
        }

        return array(
            'success' => false,
            'message' => 'Nieoczekiwana odpowiedź API',
        );
    }

    // ========== Jobs ==========

    /**
     * Pobierz listę zadań
     *
     * @param int    $limit  Limit wyników
     * @param string $status Filtr statusu
     */
    public function get_jobs($limit = 20, $status = null) {
        $endpoint = '/api/jobs?limit=' . intval($limit);
        if ($status) {
            $endpoint .= '&status_filter=' . urlencode($status);
        }
        return $this->request($endpoint);
    }

    /**
     * Pobierz szczegóły zadania
     *
     * @param string $job_id ID zadania
     */
    public function get_job($job_id) {
        return $this->request('/api/jobs/' . urlencode($job_id));
    }

    /**
     * Pobierz logi zadania
     *
     * @param string $job_id ID zadania
     * @param int    $limit  Limit logów
     */
    public function get_job_logs($job_id, $limit = 1000) {
        return $this->request('/api/jobs/' . urlencode($job_id) . '/logs?limit=' . intval($limit));
    }

    /**
     * Uruchom moduł
     *
     * @param string $module Nazwa modułu
     * @param array  $params Parametry
     */
    public function run_module($module, $params = array()) {
        return $this->request('/api/jobs/' . urlencode($module), 'POST', $params);
    }

    /**
     * Anuluj zadanie
     *
     * @param string $job_id ID zadania
     */
    public function cancel_job($job_id) {
        return $this->request('/api/jobs/' . urlencode($job_id), 'DELETE');
    }

    // ========== Skróty dla modułów ==========

    /**
     * Uruchom Stock Sync
     *
     * @param bool $force_reset Wymuś reset
     */
    public function run_stock_sync($force_reset = false) {
        return $this->run_module('stock-sync', array('force_reset' => $force_reset));
    }

    /**
     * Uruchom Product Adder
     *
     * @param int $limit Limit produktów
     */
    public function run_product_adder($limit = null) {
        $params = array();
        if ($limit !== null) {
            $params['limit'] = intval($limit);
        }
        return $this->run_module('product-adder', $params);
    }

    /**
     * Uruchom Product Deleter
     *
     * @param bool $dry_run Tryb testowy
     */
    public function run_product_deleter($dry_run = true) {
        return $this->run_module('product-deleter', array('dry_run' => $dry_run));
    }

    /**
     * Uruchom Price Updater
     */
    public function run_price_updater() {
        return $this->run_module('price-updater', array());
    }

    /**
     * Uruchom Order Forwarder
     *
     * @param bool $dry_run Tryb testowy
     */
    public function run_order_forwarder($dry_run = true) {
        return $this->run_module('order-forwarder', array('dry_run' => $dry_run));
    }

    /**
     * Uruchom Description Generator
     *
     * @param string $model Model AI
     * @param int    $limit Limit produktów
     */
    public function run_description_generator($model = null, $limit = null) {
        $params = array();
        if ($model !== null) {
            $params['model'] = $model;
        }
        if ($limit !== null) {
            $params['limit'] = intval($limit);
        }
        return $this->run_module('description-generator', $params);
    }

    // ========== Config ==========

    /**
     * Pobierz konfigurację
     */
    public function get_config() {
        return $this->request('/api/config');
    }

    /**
     * Pobierz konfigurację modułów
     */
    public function get_modules_config() {
        return $this->request('/api/config/modules');
    }

    /**
     * Pobierz konfigurację konkretnego modułu
     *
     * @param string $module_name Nazwa modułu
     */
    public function get_module_config($module_name) {
        return $this->request('/api/config/modules/' . urlencode($module_name));
    }

    /**
     * Aktualizuj konfigurację modułu
     *
     * @param string $module_name Nazwa modułu
     * @param array  $config      Konfiguracja
     */
    public function update_module_config($module_name, $config) {
        return $this->request('/api/config/modules/' . urlencode($module_name), 'PUT', $config);
    }

    /**
     * Pobierz listę dostępnych modułów
     */
    public function get_modules_list() {
        return $this->request('/api/modules');
    }

    // ========== SSE URL ==========

    /**
     * Pobierz URL dla SSE logów
     *
     * @param string $job_id ID zadania
     */
    public function get_sse_logs_url($job_id) {
        return $this->api_url . '/api/sse/logs/' . urlencode($job_id) . '?api_key=' . urlencode($this->api_key);
    }

    /**
     * Pobierz URL dla SSE dashboard
     */
    public function get_sse_dashboard_url() {
        return $this->api_url . '/api/sse/dashboard?api_key=' . urlencode($this->api_key);
    }

    // ========== Helpers ==========

    /**
     * Pobierz URL API
     */
    public function get_api_url() {
        return $this->api_url;
    }

    /**
     * Ustaw timeout
     *
     * @param int $timeout Timeout w sekundach
     */
    public function set_timeout($timeout) {
        $this->timeout = intval($timeout);
    }
}
